/*--------------------------------------------------------------------------*/
/* ALBERTA:  an Adaptive multi Level finite element toolbox using           */
/*           Bisectioning refinement and Error control by Residual          */
/*           Techniques for scientific Applications                         */
/*                                                                          */
/* file:     read_mesh.c                                                    */
/*                                                                          */
/* description:  reading data of mesh and vectors in machine independent    */
/*               and native binary formats.                                 */
/*                                                                          */
/*--------------------------------------------------------------------------*/
/*                                                                          */
/*  authors:   Alfred Schmidt                                               */
/*             Zentrum fuer Technomathematik                                */
/*             Fachbereich 3 Mathematik/Informatik                          */
/*             Univesitaet Bremen                                           */
/*             Bibliothekstr. 2                                             */
/*             D-28359 Bremen, Germany                                      */
/*                                                                          */
/*             Kunibert G. Siebert                                          */
/*             Institut fuer Mathematik                                     */
/*             Universitaet Augsburg                                        */
/*             Universitaetsstr. 14                                         */
/*             D-86159 Augsburg, Germany                                    */
/*                                                                          */
/*  http://www.mathematik.uni-freiburg.de/IAM/ALBERTA                       */
/*                                                                          */
/*  (c) by A. Schmidt and K.G. Siebert (1996-2003)                          */
/*                                                                          */
/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/
/* to_do :  number of dofs depending on node                                */
/* nach read_macro, damit dort pointer auf mel, v vorhanden sind!!!         */
/*    (fuers naechste Schreiben)                                            */
/*  error handling is terrible                                              */
/*--------------------------------------------------------------------------*/

#include <string.h>
#include "alberta.h"
#include "alberta_intern.h"

static XDR        *xdrp; 

/*
   WARNING:
   XDR routines to read/write ALBERTA types must be changed if the
   ALBERTA types change!

   current state:  REAL   = double
                   U_CHAR = unsigned char
                   S_CHAR = signed char
                   DOF    = int

   Another WARNING! (D.K.)
   XDR routines are not well documented in the "xdr" man page.
   Do not change anything unless you know what you are doing!
*/

bool_t AI_xdr_int(XDR *xdr, void *ip)
{
  FUNCNAME("AI_xdr_int");

  TEST_EXIT(sizeof(int) == 4,
	    "sizeof(int) != 4, please start hacking!");
#if HAVE_XDR_INT32_T
  return (xdr_int32_t(xdr,(int *)ip));
#elif HAVE_XDR_INT
  return (xdr_int(xdr,(int *)ip));
#endif
}

bool_t AI_xdr_REAL(XDR *xdr, void *rp)
{ 
  FUNCNAME("AI_xdr_REAL");

  TEST_EXIT(sizeof(double) == sizeof(REAL),
	    "sizeof(double) != sizeof(REAL), please start hacking!");
  return (xdr_double(xdr,(double *)rp));
}

bool_t AI_xdr_U_CHAR(XDR *xdr, void *ucp)
{ 
  return (xdr_u_char(xdr,(u_char *)ucp));
}

bool_t AI_xdr_S_CHAR(XDR *xdr, void *cp)
{ 
  return (xdr_char(xdr,(char *)cp));
}

bool_t AI_xdr_DOF(XDR *xdr, void *dp)
{ 
  FUNCNAME("AI_xdr_DOF");

  TEST_EXIT(sizeof(DOF) == sizeof(int),
	    "sizeof(DOF) != sizeof(int), please start hacking!");
  return (AI_xdr_int(xdr,(int *)dp));
}

#if 0
static int read_xdr_file(void *file, void *buffer, u_int size)
{
  return ((int)fread(buffer, 1, (size_t)size, (FILE *)file));
} 

static int write_xdr_file(void *file, void *buffer, u_int size)
{
  return (fwrite(buffer, (size_t)size, 1, (FILE *)file) == 1 ? (int)size : 0);

#endif

static FILE *file;

XDR *AI_xdr_open_file(const char *fn, enum xdr_op mode)
{
  FUNCNAME("AI_xdr_open_file");
  XDR *xdr;

  if (!(xdr = MEM_ALLOC(1,XDR)))
  { 
    ERROR("can't allocate memory for xdr pointer.\n");

    return NULL;
  }
  
  if ((file = fopen(fn, (mode == XDR_DECODE) ? "r": "w")))
  {
    xdrstdio_create(xdr, file, mode);
#if 0
    xdrrec_create(xdr, 65536, 65536, (caddr_t) file, 
                  (int (*)())read_xdr_file, (int (*)())write_xdr_file);
    
    xdr->x_op = mode;
    xdr->x_public = (caddr_t)file;
    
    if (mode == XDR_DECODE)
      xdrrec_skiprecord(xdr);
#endif
  
    return xdr;
  }
  else
  {
    ERROR("error opening xdr file.\n"); 
 
    MEM_FREE(xdr,1,XDR);

    return NULL;
  }
}


int AI_xdr_close_file(XDR *xdr)
{
  FUNCNAME("AI_xdr_close_file");
  if (!xdr)
  {
    ERROR("NULL xdr pointer.\n");
    return 0;
  }

#if 0
  if (xdr->x_op == XDR_ENCODE)
    xdrrec_endofrecord(xdr, 1);
#endif

  xdr_destroy(xdr);
 
  if (fclose(file))
    ERROR("error closing file.\n");

  MEM_FREE(xdr,1,XDR);

  return 1;
}

/*--------------------------------------------------------------------------*/

static void read_REAL(REAL *val)
{
  if(xdrp)
    AI_xdr_REAL(xdrp, val);
  else
    fread(val, sizeof(REAL), 1, file);
}

static void read_int(int *val)
{
  if(xdrp)
    AI_xdr_int(xdrp, val);
  else
    fread(val, sizeof(int), 1, file);
}

#if 0
/* NEVER use this. If you need 64 bits, then use xdr_int64_t() */
static void read_long(long *val)
{
  if(xdrp)
    xdr_long(xdrp, val);
  else
    fread(val, sizeof(long int), 1, file);
}
#endif

static void read_string(char **string, int read_size, int strileng)
{
  if(read_size) {
    read_int(&strileng);

    if(strileng)
      *string = (char *)MEM_ALLOC(strileng+1, char);
  }    

  if(xdrp)
    xdr_string(xdrp, string, strileng+1);
  else
    fread(*string, sizeof(char), strileng+1, file);
}

static void read_vector(void *start, int n, size_t size, xdrproc_t xdrproc)
{
  if(xdrp)
    xdr_vector(xdrp, (char *)start, n, size, xdrproc);
  else
    fread(start, size, n, file);
}

static void read_U_CHAR(U_CHAR *val)
{
  if(xdrp)
    AI_xdr_U_CHAR(xdrp, val);
  else
    fread(val, sizeof(U_CHAR), 1, file);
}

/*--------------------------------------------------------------------------*/

static DOF_ADMIN  *admin = nil;
static MESH       *mesh = nil;

static int        n_vert_dofs;
static DOF        **vert_dofs;

static int        n_edge_dofs;
static DOF        **edge_dofs;

static int        n_face_dofs;
static DOF        **face_dofs;

/*--------------------------------------------------------------------------*/

/****************************************************************************/
/* read_dofs(mesh, dof_ptr, type): Read a DOF vector of type "type" from    */
/* the file. Allocate space for nonzero DOF_entries (which are overwritten  */
/* with the indices from the file).                                         */
/****************************************************************************/

static void read_dofs(MESH *mesh, DOF **dof_ptr, int type)
{
  int        n_dof = mesh->n_dof[type];
  DOF        temp_dofs[n_dof];
  DOF_ADMIN *admin;
  int        i, j, n, n0;

  read_vector(temp_dofs, n_dof, sizeof(DOF), (xdrproc_t)AI_xdr_DOF); 
  *dof_ptr = AI_get_dof_memory(mesh, type);

  for (i = 0; i < mesh->n_dof_admin; i++) {
    admin = mesh->dof_admin[i];
    
    n  = admin->n_dof[type];
    n0 = admin->n0_dof[type];
    
    TEST_EXIT(n+n0 <= n_dof,"dof_admin \"%s\": n=%d, n0=%d too large: ndof=%d\n", admin->name, n, n0, n_dof);
    
    for (j = 0; j < n; j++) {
      (*dof_ptr)[n0+j] = temp_dofs[n0+j];

      if(temp_dofs[n0+j] >= 0) 
	get_dof_index(admin); /* Reserve new DOF but forget the index. */
    }
  }
  
  return;
}

/*--------------------------------------------------------------------------*/

static EL *read_el_recursive(int dim, EL *parent)
{
  FUNCNAME("read_el_recursive");
  int    i, j, n, node0;
  EL     *el;
  U_CHAR uc, nc;

  el = get_element(mesh);
  mesh->n_hier_elements++;

#if ALBERTA_DEBUG
  el->index = mesh->n_hier_elements;
#endif
 
  read_U_CHAR(&uc); 

  if(dim > 1) {
    read_U_CHAR(&nc);  
    if (nc) {
	el->new_coord = get_real_d(mesh);
	
	read_vector(el->new_coord, DIM_OF_WORLD, sizeof(REAL), 
		    (xdrproc_t)AI_xdr_REAL);
    }
    else 
      el->new_coord = nil;
  }

  if (mesh->n_dof[VERTEX] > 0) {
    node0 = mesh->node[VERTEX];
    for (i = 0; i < N_VERTICES(dim); i++) {
      read_int(&j);

      TEST_EXIT(j < n_vert_dofs,
	"vert_dofs index too large: %d >= %d\n", j, n_vert_dofs);
      el->dof[node0 + i] = vert_dofs[j];
    }
  }

  if (dim > 1 && mesh->n_dof[EDGE] > 0) {
    node0 = mesh->node[EDGE];
    for (i = 0; i < N_EDGES(dim); i++) {
      read_int(&j);
      
      TEST_EXIT(j < n_edge_dofs,
	"edge_dofs index too large: %d >= %d\n", j, n_edge_dofs);
      if(j >= 0)
	el->dof[node0 + i] = edge_dofs[j];
    }
  }
  
  if (dim == 3 && (n = mesh->n_dof[FACE]) > 0) {
    node0 = mesh->node[FACE];
    for (i = 0; i < N_FACES_3D; i++) {
      read_int(&j);
      
      TEST_EXIT(j < n_face_dofs,
	"face_dofs index too large: %d >= %d\n", j, n_face_dofs);
      if(j >= 0)
	el->dof[node0 + i] = face_dofs[j];
    }
  }
  
  if ((n = mesh->n_dof[CENTER]) > 0) {
    node0 = mesh->node[CENTER];
    
    read_dofs(mesh, el->dof + node0, CENTER);
  }

  if (uc) {
    el->child[0] = read_el_recursive(dim, el);
    el->child[1] = read_el_recursive(dim, el);
  }
  else
    mesh->n_elements++;

  return(el);
}

/*--------------------------------------------------------------------------*/

static void read_dof_admins(MESH *mesh)
{
  FUNCNAME("read_dof_admins");
  int      i, n_dof_admin, iadmin, used_count;
  int      n_dof_el, n_dof[N_NODE_TYPES], n_node_el, node[N_NODE_TYPES];
  int      a_n_dof[N_NODE_TYPES];
  U_CHAR   pcd;
  char     *name;

 
  read_int(&n_dof_el);
  read_vector(n_dof, N_NODE_TYPES, sizeof(int), (xdrproc_t)AI_xdr_int);
  read_int(&n_node_el);
  read_vector(node, N_NODE_TYPES, sizeof(int), (xdrproc_t)AI_xdr_int);
  /* use data later for check */
  
  read_int(&n_dof_admin); 
  for (iadmin = 0; iadmin < n_dof_admin; iadmin++)
  {
    read_vector(a_n_dof, N_NODE_TYPES, sizeof(int), (xdrproc_t)AI_xdr_int); 
    read_int(&used_count);

    read_string(&name, true, 0);  

    read_U_CHAR(&pcd);

    admin = AI_get_dof_admin(mesh, name, a_n_dof);
    admin->preserve_coarse_dofs = pcd;

    MEM_FREE(name, strlen(name)+1, char);

    if (used_count > 0) enlarge_dof_lists(admin, used_count);
  } /* end for (iadmin) */


  for(i = 0; i < N_NODE_TYPES; i++)
    if(mesh->n_dof[i])
      AI_get_dof_list(mesh, i);

  AI_get_dof_ptr_list(mesh);

  TEST(mesh->n_dof_el == n_dof_el,"wrong n_dof_el: %d %d\n", 
				   mesh->n_dof_el, n_dof_el);
  for (i = 0; i < N_NODE_TYPES; i++)
    TEST(mesh->n_dof[i] == n_dof[i],"wrong n_dof[%d]: %d %d\n",
				     i, mesh->n_dof[i], n_dof[i]);
  TEST(mesh->n_node_el == n_node_el,"wrong n_node_el: %d %d\n",
				     mesh->n_node_el, n_node_el);
  for (i = 0; i < N_NODE_TYPES; i++)
    TEST(mesh->node[i] == node[i],"wrong node[%d]: %d %d\n",
                                   i, mesh->node[i], node[i]); 
  return;
}

/*--------------------------------------------------------------------------*/

static MESH *read_mesh_master(const int write_xdr, const char *fn,
			      REAL *timeptr,
		      NODE_PROJECTION *(*n_proj)(MESH *, MACRO_EL *, int))
{
  FUNCNAME("read_mesh_master");
  MACRO_EL       *mel;
  int            i, j, n;
  REAL_D         *v, x_min, x_max;
  int            neigh_i[N_NEIGH_MAX];
  char           *name, *s;
  size_t         length;
  int            dim, iDIM_OF_WORLD, ne, nv;
  REAL           time, diam[DIM_OF_WORLD];
  int            n_vert, n_elements, n_hier_elements;
  int            n_edges;
  int            vert_i[N_VERTICES_MAX];
  static int     funccount=0;
  int            n_faces, max_edge_neigh;

  length = MAX(strlen(ALBERTA_VERSION)+1,5);  /* length with terminating \0 */ 
  s = MEM_ALLOC(length, char);

  if(write_xdr) {
    if (!(xdrp = AI_xdr_open_file(fn, XDR_DECODE))) {
      ERROR("Cannot open XDR file '%s'\n",fn);
      goto error_exit;
    }
  }
  else if (!(file=fopen(fn,"rb"))) {
    ERROR("Cannot open file '%s'\n", fn);
    goto error_exit;
  }

  read_string(&s, false, strlen(ALBERTA_VERSION));
   
  if (strcmp(s, ALBERTA_VERSION))
  {
    ERROR("Invalid file id: \"%s\", expected \"%s\"\n",s, ALBERTA_VERSION);
    goto error_exit;
  } 
  
  read_int(&dim);
  if (dim > DIM_OF_WORLD) 
  { 
    ERROR("dim==%d is greater than DIM_OF_WORLD==%d!\n", dim, DIM_OF_WORLD);
    goto error_exit; 
  } 

  read_int(&iDIM_OF_WORLD); 
  if (iDIM_OF_WORLD != DIM_OF_WORLD) 
  { 
    ERROR("wrong DIM_OF_WORLD %d. abort.\n", iDIM_OF_WORLD); 
    goto error_exit; 
  } 

  read_REAL(&time);
  if (timeptr) *timeptr = time;

  

  read_int(&i);                         /* length without terminating \0 */
  if(i) {
    name = MEM_ALLOC(i+1, char);
    read_string(&name, false, i);  
  }
  else {
    funccount++;
    i=100;
    name = MEM_ALLOC(i+1, char);
    sprintf(name, "READ_MESH%d", funccount);
  }

  
  read_int(&n_vert);
  if(dim > 1)
    read_int(&n_edges);

  read_int(&n_elements);
  read_int(&n_hier_elements);


  if(dim == 3) {
    read_int(&n_faces);
    read_int(&max_edge_neigh);
  }

  read_vector(diam, DIM_OF_WORLD, sizeof(REAL), (xdrproc_t)AI_xdr_REAL);
  

  mesh = GET_MESH(dim, name, nil, nil);
  read_dof_admins(mesh);


  MEM_FREE(name, i+1, char);

  read_int(&n_vert_dofs);   

  if (n_vert_dofs > 0) 
  {
    vert_dofs = MEM_ALLOC(n_vert_dofs, DOF *); 
    n = mesh->n_dof[VERTEX]; 
    for (i = 0; i < n_vert_dofs; i++) 
    {
      vert_dofs[i] = get_dof(mesh, VERTEX); 
      
      read_vector(vert_dofs[i], n, sizeof(DOF), (xdrproc_t)AI_xdr_DOF); 
    }
  } 

  if(dim > 1) {
    read_int(&n_edge_dofs);
    
    if (n_edge_dofs > 0) {
      edge_dofs = MEM_ALLOC(n_edge_dofs, DOF *); 
      
      for (i = 0; i < n_edge_dofs; i++) 
	read_dofs(mesh, edge_dofs + i, EDGE);
    } 
  }

  if(dim == 3) { 
    read_int(&n_face_dofs);
    
    if (n_face_dofs > 0) 
      {
	face_dofs = MEM_ALLOC(n_face_dofs, DOF *); 

	for (i = 0; i < n_face_dofs; i++) 
	  read_dofs(mesh, face_dofs + i, FACE);
      } 
  }

  read_int(&ne);
  read_int(&nv); 

  ((MESH_MEM_INFO *)mesh->mem_info)->count = nv;
  v = ((MESH_MEM_INFO *)mesh->mem_info)->coords = MEM_ALLOC(nv, REAL_D);

  for (i = 0; i < nv; i++) 
     read_vector(v[i], DIM_OF_WORLD, sizeof(REAL), (xdrproc_t)AI_xdr_REAL);

  for (j = 0; j < DIM_OF_WORLD; j++) 
  { 
    x_min[j] =  1.E30; 
    x_max[j] = -1.E30; 
  } 

  for (i = 0; i < nv; i++)  
    for (j = 0; j < DIM_OF_WORLD; j++) 
    { 
      x_min[j] = MIN(x_min[j], v[i][j]); 
      x_max[j] = MAX(x_max[j], v[i][j]); 
    } 

  for (j = 0; j < DIM_OF_WORLD; j++) 
    mesh->diam[j] = x_max[j] - x_min[j]; 


  mel  = MEM_CALLOC(ne, MACRO_EL); 

  mesh->n_macro_el = ne; 
  mesh->macro_els = mel; 


  for (n = 0; n < ne; n++) 
  { 
    mel[n].index = n; 

    read_vector(vert_i, N_VERTICES(dim), sizeof(int), (xdrproc_t)AI_xdr_int);
    
    for (i = 0; i < N_VERTICES(dim); i++) 
    { 
      if ((vert_i[i] >= 0) && (vert_i[i] < nv)) 
	mel[n].coord[i] = (REAL *)(v + (vert_i[i])); 
      else 
	mel[n].coord[i] = nil; 
    } 

    
    read_vector(mel[n].vertex_bound,N_VERTICES(dim),sizeof(S_CHAR),
		(xdrproc_t)AI_xdr_S_CHAR);
    
    if(dim == 2) {
      read_vector(mel[n].edge_bound, N_EDGES_2D, sizeof(S_CHAR),
		  (xdrproc_t)AI_xdr_S_CHAR);
    }

    if(dim == 3) {
      read_vector(mel[n].face_bound, N_FACES_3D, sizeof(S_CHAR), 
		  (xdrproc_t)AI_xdr_S_CHAR);
      read_vector(mel[n].edge_bound, N_EDGES_3D, sizeof(S_CHAR), 
		  (xdrproc_t)AI_xdr_S_CHAR);
    }

    read_vector(neigh_i, N_NEIGH(dim), sizeof(int), (xdrproc_t)AI_xdr_int);
   
    if(n_proj)
      mel[n].projection[0] = n_proj(mesh, mel + n, 0);
    for (i = 0; i < N_NEIGH(dim); i++)  { 
      if(n_proj && (dim > 1))
	mel[n].projection[i+1] = n_proj(mesh, mel + n, i+1);
      
      if ((neigh_i[i] >= 0) && (neigh_i[i] < ne)) 
	mel[n].neigh[i] = mel + (neigh_i[i]); 
      else 
	mel[n].neigh[i] = nil; 
    } 

    read_vector(mel[n].opp_vertex, N_NEIGH(dim), sizeof(U_CHAR), 
		(xdrproc_t)AI_xdr_U_CHAR);

#if DIM_OF_WORLD == 3
    if(dim == 3) {
      read_U_CHAR(&(mel[n].el_type));            
      mel[n].orientation = AI_get_orientation(&mel[n]);
    }
#endif

    mel[n].el = read_el_recursive(dim, nil); 
  }

/****************************************************************************/
/* The present mechanism only reads DOFs from the file which were used by   */
/* at least one DOF_ADMIN. The missing element DOF pointers are filled by   */
/* this routine (in memory.c).                                              */
/****************************************************************************/
  if(dim > 0)
    AI_fill_missing_dofs(mesh);

  if (n_elements != mesh->n_elements) 
  { 
    ERROR("n_elements != mesh->n_elements.\n"); 
    goto error_exit; 
  } 

  if (n_hier_elements != mesh->n_hier_elements) 
  { 
    ERROR("n_hier_elements != mesh->n_hier_elements.\n"); 
    goto error_exit; 
  } 
  

  if (mesh->n_dof[VERTEX])
  {
    if (n_vert != n_vert_dofs) 
    { 
      ERROR("n_vertices == %d != %d == n_vert_dofs.\n", n_vert, n_vert_dofs); 
      mesh->n_vertices = n_vert_dofs; 
      goto error_exit; 
    }
  }
  mesh->n_vertices = n_vert;

  if (dim > 1)
    mesh->n_edges = n_edges;

  if(dim == 3) {
    mesh->n_faces = n_faces;
    
    mesh->max_edge_neigh = max_edge_neigh;
  }

  for (i=0; i<DIM_OF_WORLD; i++) 
  {
    if (ABS(mesh->diam[i]-diam[i]) > (mesh->diam[i]/10000.0))
    {
      ERROR("diam[%i] != mesh->diam[%i].\n",i,i);
      /* goto error_exit; */
    }
  }
 
  /* Read the magic cookie. */
  read_int(&mesh->cookie);

  read_string(&s, false, 4);  
  if (strncmp(s, "EOF.", 4))                    /* file end marker */
  { 
    ERROR("no FILE END MARK.\n"); 
    goto error_exit; 
  } 
  else
  {
    MSG("File %s read.\n",fn);
  }

error_exit:
  MEM_FREE(s, length, char);
  
  if(xdrp)
    AI_xdr_close_file(xdrp);
  else
    fclose(file);

  xdrp = nil;
  file = nil;

  return(mesh);
}

MESH *read_mesh_xdr(const char *fn, REAL *timeptr,
		    NODE_PROJECTION *(*n_proj)(MESH *, MACRO_EL *, int))
{
  return read_mesh_master(true, fn, timeptr, n_proj);
}

MESH *read_mesh(const char *fn, REAL *timeptr,
		NODE_PROJECTION *(*n_proj)(MESH *, MACRO_EL *, int))
{
  return read_mesh_master(false, fn, timeptr, n_proj);
}

/*--------------------------------------------------------------------------*/
/* read DOF vectors of various types                                        */
/*--------------------------------------------------------------------------*/

typedef DOF_REAL_VEC DOF_VEC;

static const DOF_ADMIN *read_dof_vec_master(const int read_xdr,
					    const char *fn, DOF_VEC *dv,
					    const char *dofvectype,
					    MESH *mesh, FE_SPACE *fe_space)
{
  FUNCNAME("read_dof_vec_master");
  int             i, last;
  int             n_dof[N_NODE_TYPES];
  int             cookie;
  const DOF_ADMIN *admin = nil;
  const BAS_FCTS  *bas_fcts;
  char            *name, *s;
  size_t          length;  
  U_CHAR          pcd;

  TEST_EXIT(mesh,"no mesh given\n");
 
  if(read_xdr) {
    if (!(xdrp = AI_xdr_open_file(fn, XDR_DECODE))) {
      ERROR("Cannot open XDR file '%s'\n",fn);
      return nil;
    }
  }
  else if(!(file = fopen(fn, "rb"))) {
    ERROR("Cannot open file '%s'\n",fn);
    return nil;
  }

  length = 17;          /* length of dofvectype with terminating \0 */
  
  s=MEM_ALLOC(length, char);
  read_string(&s, false, 16);  
  
  if (strncmp(s, dofvectype, 12))
  {
    ERROR("invalid file id; %s\n", s);
    goto error_exit;
  }

  read_int(&last);
  name = MEM_ALLOC(last+1, char);
  read_string(&name, false, last);  

  read_U_CHAR(&pcd);

  dv->name = name; 
  read_vector(n_dof, N_NODE_TYPES, sizeof(int), (xdrproc_t)AI_xdr_int);  

  read_int(&last);                        /* length of BAS_FCTS name */ 
  
  if (last)
  {
    name = MEM_ALLOC(last+1, char);
    read_string(&name, false, last);    

    if (fe_space && (bas_fcts = fe_space->bas_fcts)) 
    {
      if (strcmp(bas_fcts->name, name)) 
      {
	ERROR("invalid name %s is not given fe_space->bas_fcts->name %s\n",
	      name, bas_fcts->name);
      }
    }
    else 
    { /* no given fe_space or no bas_fcts in given fe_space */
      TEST_EXIT(bas_fcts = get_bas_fcts(name),
	"cannot get bas_fcts <%s>\n", name);

      if (fe_space) 
      {                             /* use given fe_space */
	fe_space->bas_fcts = bas_fcts;
      }
      else 
      {                                     /* create new fe_space */
	TEST_EXIT(fe_space =
		  (FE_SPACE *)get_fe_space(mesh, name, n_dof, bas_fcts, pcd),
	  "cannot get fe_space for bas_fcts <%s>\n", name);
      }
    }

    for (i = 0; i < N_NODE_TYPES; i++) 
    {
      TEST_EXIT(n_dof[i] == bas_fcts->n_dof[i],
	"wrong n_dof in bas_fcts <%s>\n", name);
    }
  }
  else 
  {  /* no bas_fcts.name in file */
    if (fe_space) 
    {                               /* use given fe_space */
      TEST_EXIT(admin = fe_space->admin,"no fe_space->admin");
      for (i = 0; i < N_NODE_TYPES; i++) 
      {
	TEST_EXIT(n_dof[i] == admin->n_dof[i],
	  "wrong n_dof in admin <%s>\n", NAME(admin));
      }
    }
    else 
    {                                       /* create new fe_space */
      TEST_EXIT(fe_space =
		(FE_SPACE *) get_fe_space(mesh, nil, n_dof, nil, pcd),
	"cannot get fe_space for given n_dof\n");
      TEST_EXIT(admin = fe_space->admin,"no admin in new fe_space\n");
      for (i = 0; i < N_NODE_TYPES; i++) 
      {
	TEST_EXIT(n_dof[i] == admin->n_dof[i],
	  "wrong n_dof in admin <%s>\n", NAME(admin));
      }
    }
  }
  TEST_EXIT(fe_space,"still no fe_space\n");
  dv->fe_space = fe_space;
  TEST_EXIT(admin = fe_space->admin,"still no admin\n");

  dof_compress(mesh);
 
  read_int(&last);
  if (last != admin->size_used)
  {
    ERROR("size of dof vector `%s' == %d does not fit to size_used == %d in admin `%s'\n", dv->name, last, admin->size_used, admin->name);
    ERROR_EXIT("can not read incompatible data\n");
  }

  if (last)
  {
    dv->size = last;
    
    if (!strncmp(dofvectype, "DOF_REAL_VEC    ", 12))
    {
      dv->vec = ((REAL *) alberta_alloc((size_t)(last*sizeof(double)), funcName,
	                                __FILE__, __LINE__));
      read_vector(dv->vec, last, sizeof(REAL), (xdrproc_t)AI_xdr_REAL);  
    }
    else if (!strncmp(dofvectype, "DOF_REAL_D_VEC  ", 12))
    {
      dv->vec = 
       ((REAL *) alberta_alloc((size_t)(last*DIM_OF_WORLD*sizeof(REAL)), 
                              funcName, __FILE__, __LINE__));
      read_vector(dv->vec, last*DIM_OF_WORLD, sizeof(REAL), 
		  (xdrproc_t)AI_xdr_REAL);
    }
    else if (!strncmp(dofvectype, "DOF_INT_VEC     ", 12))
    {
      dv->vec = ((REAL *) alberta_alloc((size_t)(last*sizeof(int)), funcName,
	                                __FILE__, __LINE__));
      read_vector(dv->vec, last, sizeof(int), (xdrproc_t)AI_xdr_int);  
    }
    else if (!strncmp(dofvectype, "DOF_SCHAR_VEC   ", 12))
    {
      dv->vec = ((REAL *) alberta_alloc((size_t)(last*sizeof(char)), funcName,
	                                __FILE__, __LINE__));
      read_vector(dv->vec, last, sizeof(S_CHAR), (xdrproc_t)AI_xdr_S_CHAR);  
    }
    else if (!strncmp(dofvectype, "DOF_UCHAR_VEC   ", 12))
    {
      dv->vec = ((REAL *) alberta_alloc((size_t)(last*sizeof(unsigned char)), 
                                       funcName, __FILE__, __LINE__));
      read_vector(dv->vec, last, sizeof(U_CHAR), (xdrproc_t)AI_xdr_U_CHAR);  
    }
    else
      ERROR("Invalid file id '%s'\n", dofvectype);
  }      
  else
  {
    ERROR("empty dof vector\n");
    dv->size = 0;
    dv->vec = nil;
  }

  /* Read the magic cookie. */
  read_int(&cookie);

  if (cookie != mesh->cookie)
    WARNING("Mesh and DOF vector do not seem to match!\n");

  read_string(&s, false, 4);  
  if (strncmp(s, "EOF.", 4))                    /* file end marker */
  { 
    ERROR("no FILE END MARK.\n"); 
    goto error_exit; 
  } 
  else
  {
    MSG("File '%s' read.\n",fn);
  }

 error_exit:
  MEM_FREE(s,length,char);

  if(xdrp)
    AI_xdr_close_file(xdrp);
  else
    fclose(file);

  xdrp = nil;
  file = nil;

  return(admin);
}
/*--------------------------------------------------------------------------*/

DOF_REAL_VEC *read_dof_real_vec_xdr(const char *fn, MESH *mesh, FE_SPACE
				    *fe_space)
{
  DOF_REAL_VEC    *dv;
  const DOF_ADMIN *admin;

  dv = get_dof_real_vec(fn, nil);
  
  admin = read_dof_vec_master(true, fn, (DOF_VEC *)dv, "DOF_REAL_VEC    ",
			      mesh, fe_space);

  if (admin) AI_add_dof_real_vec_to_admin(dv, (DOF_ADMIN *)admin);
  return(dv);
}

DOF_REAL_VEC *read_dof_real_vec(const char *fn, MESH *mesh, FE_SPACE
				    *fe_space)
{
  DOF_REAL_VEC    *dv;
  const DOF_ADMIN *admin;

  dv = get_dof_real_vec(fn, nil);
  
  admin = read_dof_vec_master(false, fn, (DOF_VEC *)dv, "DOF_REAL_VEC    ",
			      mesh, fe_space);

  if (admin) AI_add_dof_real_vec_to_admin(dv, (DOF_ADMIN *)admin);
  return(dv);
}

/*--------------------------------------------------------------------------*/

DOF_REAL_D_VEC *read_dof_real_d_vec(const char *fn, MESH *mesh, 
				    FE_SPACE *fe_space)
{
  DOF_REAL_D_VEC  *dv;
  const DOF_ADMIN *admin;

  dv = get_dof_real_d_vec(fn, nil);
 
  admin = read_dof_vec_master(false, fn, (DOF_VEC *)dv, "DOF_REAL_D_VEC  ",
			      mesh, fe_space);
 
  if (admin) AI_add_dof_real_d_vec_to_admin(dv, (DOF_ADMIN *)admin);
  return(dv);
}

DOF_REAL_D_VEC *read_dof_real_d_vec_xdr(const char *fn, MESH *mesh, 
					FE_SPACE *fe_space)
{
  DOF_REAL_D_VEC  *dv;
  const DOF_ADMIN *admin;

  dv = get_dof_real_d_vec(fn, nil);
 
  admin = read_dof_vec_master(true, fn, (DOF_VEC *)dv, "DOF_REAL_D_VEC  ",
			      mesh, fe_space);
 
  if (admin) AI_add_dof_real_d_vec_to_admin(dv, (DOF_ADMIN *)admin);
  return(dv);
}

/*--------------------------------------------------------------------------*/

DOF_INT_VEC *read_dof_int_vec(const char *fn, MESH *mesh, 
			      FE_SPACE *fe_space)
{
  DOF_INT_VEC     *dv;
  const DOF_ADMIN *admin;

  dv = get_dof_int_vec(fn, nil);
  
  admin = read_dof_vec_master(false, fn, (DOF_VEC *)dv, "DOF_INT_VEC     ",
			      mesh, fe_space);

  if (admin) AI_add_dof_int_vec_to_admin(dv, (DOF_ADMIN *)admin);
  return(dv);
}

DOF_INT_VEC *read_dof_int_vec_xdr(const char *fn, MESH *mesh, 
				  FE_SPACE *fe_space)
{
  DOF_INT_VEC     *dv;
  const DOF_ADMIN *admin;

  dv = get_dof_int_vec(fn, nil);
  
  admin = read_dof_vec_master(true, fn, (DOF_VEC *)dv, "DOF_INT_VEC     ",
			      mesh, fe_space);

  if (admin) AI_add_dof_int_vec_to_admin(dv, (DOF_ADMIN *)admin);
  return(dv);
}

/*--------------------------------------------------------------------------*/

DOF_SCHAR_VEC *read_dof_schar_vec(const char *fn, MESH *mesh,
				  FE_SPACE *fe_space)
{
  DOF_SCHAR_VEC   *dv;
  const DOF_ADMIN *admin;

  dv = get_dof_schar_vec(fn, nil);
 
  admin = read_dof_vec_master(false, fn, (DOF_VEC *)dv, "DOF_SCHAR_VEC   ",
			      mesh, fe_space);

  if (admin) AI_add_dof_schar_vec_to_admin(dv, (DOF_ADMIN *)admin);
  return(dv);
}

DOF_SCHAR_VEC *read_dof_schar_vec_xdr(const char *fn, MESH *mesh,
				      FE_SPACE *fe_space)
{
  DOF_SCHAR_VEC   *dv;
  const DOF_ADMIN *admin;

  dv = get_dof_schar_vec(fn, nil);
 
  admin = read_dof_vec_master(true, fn, (DOF_VEC *)dv, "DOF_SCHAR_VEC   ",
			      mesh, fe_space);

  if (admin) AI_add_dof_schar_vec_to_admin(dv, (DOF_ADMIN *)admin);
  return(dv);
}

/*--------------------------------------------------------------------------*/

DOF_UCHAR_VEC *read_dof_uchar_vec(const char *fn, MESH *mesh, 
				  FE_SPACE *fe_space)
{
  DOF_UCHAR_VEC   *dv;
  const DOF_ADMIN *admin;

  dv = get_dof_uchar_vec(fn, nil);
 
  admin = read_dof_vec_master(false, fn, (DOF_VEC *)dv, "DOF_UCHAR_VEC   ",
			      mesh, fe_space);

  if (admin) AI_add_dof_uchar_vec_to_admin(dv, (DOF_ADMIN *)admin);
  return(dv);
}

DOF_UCHAR_VEC *read_dof_uchar_vec_xdr(const char *fn, MESH *mesh, 
				      FE_SPACE *fe_space)
{
  DOF_UCHAR_VEC   *dv;
  const DOF_ADMIN *admin;

  dv = get_dof_uchar_vec(fn, nil);
 
  admin = read_dof_vec_master(true, fn, (DOF_VEC *)dv, "DOF_UCHAR_VEC   ",
			      mesh, fe_space);

  if (admin) AI_add_dof_uchar_vec_to_admin(dv, (DOF_ADMIN *)admin);
  return(dv);
}

/*--------------------------------------------------------------------------*/
