/*--------------------------------------------------------------------------*/
/* ALBERTA:  an Adaptive multi Level finite element toolbox using           */
/*           Bisectioning refinement and Error control by Residual          */
/*           Techniques for scientific Applications                         */
/*--------------------------------------------------------------------------*/
/*                                                                          */
/* file:     parametric.c                                                   */
/*                                                                          */
/*                                                                          */
/* description: Support for parametric elements.                            */
/*--------------------------------------------------------------------------*/
/*                                                                          */
/*  authors:   Daniel Koester                                               */
/*             Institut fuer Mathematik                                     */
/*             Universitaet Augsburg                                        */
/*             Universitaetsstr. 14                                         */
/*             D-86159 Augsburg, Germany                                    */
/*                                                                          */
/*  http://www.alberta-fem.de                                               */
/*                                                                          */
/*  (c) by D. Koester (2005)                                                */
/*--------------------------------------------------------------------------*/

#include "alberta.h"
#include "alberta_intern.h"

typedef struct lagrange_param_data LAGRANGE_PARAM_DATA;
struct lagrange_param_data
{
  int              i_am_affine;
  NODE_PROJECTION *n_proj;
  DOF_REAL_D_VEC  *coords;
  DOF_UCHAR_VEC   *touched_coords;
  int              n_local_coords;
  REAL_D          *local_coords;
};

/*--------------------------------------------------------------------------*/
/* param_init_element: returns true iff we are on a parametric element.     */
/*--------------------------------------------------------------------------*/

static int param_init_element(const EL_INFO *el_info,
			      const PARAMETRIC *parametric)
{
  /* FUNCNAME("param_init_element"); */
  int                  result = 0, i;
  LAGRANGE_PARAM_DATA *data = (LAGRANGE_PARAM_DATA *)parametric->data;
  int                  n_local_coords = data->n_local_coords;
  DOF_REAL_D_VEC      *coords = data->coords;
  REAL_D              *local_coords = data->local_coords;


  if(parametric->not_all) {
    DOF_UCHAR_VEC       *touched_coords = data->touched_coords;
    const U_CHAR        *local_touched;

    local_touched = data->coords->fe_space->bas_fcts->get_uchar_vec
      (el_info->el,touched_coords,nil);

    for(i = 0; i < n_local_coords; i++)
      if(local_touched[i]) {
	result = 1;
	break;
      }
  }
  else
    result = 1;

  data->coords->fe_space->bas_fcts->get_real_d_vec(el_info->el, coords,
						   local_coords);

  data->i_am_affine = 1-result;
  return result;
}

static void param_coord_to_world(const EL_INFO *el_info, const QUAD *quad, 
				 int N, const REAL lambda[][N_LAMBDA],
				 REAL_D *world)
{
  int i, iq, n;
  LAGRANGE_PARAM_DATA *data = 
    (LAGRANGE_PARAM_DATA *)el_info->mesh->parametric->data;
  REAL_D *local_coords      = data->local_coords;
  const BAS_FCTS *bas_fcts  = data->coords->fe_space->bas_fcts;

  if (quad) {
    static const QUAD_FAST *quad_fast = nil;
    REAL   **phi;

    if (!quad_fast  ||  quad_fast->quad != quad || 
	!(quad_fast->init_flag|INIT_PHI)) 
      quad_fast = get_quad_fast(bas_fcts, quad, INIT_PHI);

    phi = quad_fast->phi;

    for (iq = 0; iq < quad->n_points; iq++) {
      SET_DOW(0.0, world[iq]);

      for (i = 0; i < quad_fast->n_bas_fcts; i++) {
	for (n = 0; n < DIM_OF_WORLD; n++)
	  world[iq][n] += local_coords[i][n] * phi[iq][i];
      }
    }
  }
  else {
    BAS_FCT  **phi = bas_fcts->phi;
    REAL       phi_lambda;

    for (iq = 0; iq < N; iq++) {
      SET_DOW(0.0, world[iq]);

      for (i = 0; i < bas_fcts->n_bas_fcts; i++) {
	phi_lambda = (*phi[i])(lambda[iq]);

	for (n = 0; n < DIM_OF_WORLD; n++)
	  world[iq][n] += local_coords[i][n]*phi_lambda;
      }
    }
  }

  return;
}

#include "../1d/parametric_1d.c"
#if DIM_OF_WORLD > 1
#include "../2d/parametric_2d.c"
#if DIM_OF_WORLD > 2
#include "../3d/parametric_3d.c"
#endif
#endif

#define MAX_DEG 2

static const PARAMETRIC *all_lagrange[DIM_OF_WORLD * MAX_DEG] = 
  {&lagrange_parametric1_1d,
   &lagrange_parametric2_1d 
#if DIM_OF_WORLD > 1
   ,&lagrange_parametric1_2d,
   &lagrange_parametric2_2d
#if DIM_OF_WORLD > 2
   ,&lagrange_parametric1_3d,
   &lagrange_parametric2_3d
#endif
#endif
  };

static void (*all_refine_interpol[DIM_OF_WORLD * MAX_DEG])
     (DOF_REAL_D_VEC *, RC_LIST_EL *, int) =
{refine_interpol1_1d,
 refine_interpol2_1d
#if DIM_OF_WORLD > 1
 ,refine_interpol1_2d,
 refine_interpol2_2d
#if DIM_OF_WORLD > 2
 ,refine_interpol1_3d,
 refine_interpol2_3d
#endif
#endif
};

static void (*all_coarse_interpol[DIM_OF_WORLD * MAX_DEG])
     (DOF_REAL_D_VEC *, RC_LIST_EL *, int) =
{nil,
 coarse_interpol2_1d
#if DIM_OF_WORLD > 1
 ,nil,
 coarse_interpol2_2d
#if DIM_OF_WORLD > 2
 ,nil,
 coarse_interpol2_3d
#endif
#endif
};

static void (*all_fill_coords[DIM_OF_WORLD * MAX_DEG])
     (LAGRANGE_PARAM_DATA *data) =
{fill_coords1_1d,
 fill_coords2_1d
#if DIM_OF_WORLD > 1
 ,fill_coords1_2d,
 fill_coords2_2d
#if DIM_OF_WORLD > 2
 ,fill_coords1_3d,
 fill_coords2_3d
#endif
#endif
};


static void inherit_lagrange_parametric(MESH *slave)
{
  FUNCNAME("inherit_lagrange_parametric");
  MESH                *master;
  PARAMETRIC          *m_parametric;
  LAGRANGE_PARAM_DATA *m_data;
  DOF_INT_VEC         *dof_connection;
  DOF_REAL_D_VEC      *m_coords, *s_coords;
  DOF_UCHAR_VEC       *m_t_coords, *s_t_coords;
  REAL_D              *m_coord_vec, *s_coord_vec;
  U_CHAR              *m_t_coord_vec, *s_t_coord_vec;
  int                 *master_dofs;

  TEST_EXIT(slave, "No slave mesh given!\n");

  master = ((MESH_MEM_INFO *)slave->mem_info)->master;
  TEST_EXIT(master, "'%s' is not a slave mesh!\n", NAME(slave));

  m_parametric = master->parametric;
  TEST_EXIT(m_parametric, "'%s' is not a parametric mesh!\n", NAME(master));

  m_data = (LAGRANGE_PARAM_DATA *)m_parametric->data;
  
  /* Turn the slave mesh into a parametric mesh. */

  use_lagrange_parametric(slave, m_data->coords->fe_space->bas_fcts->degree,
			  m_data->n_proj, m_parametric->not_all);

  /* Finally, transfer coordinates and touched vertices. */
  /* Touched vertices vector is only used for strategy > 0! */

  m_coords = get_lagrange_coords(master);
  s_coords = get_lagrange_coords(slave);
  m_coord_vec = m_coords->vec;
  s_coord_vec = s_coords->vec;

  dof_connection = get_dof_int_vec("DOF connection", s_coords->fe_space);
  get_slave_dof_mapping(m_coords->fe_space, dof_connection);

  master_dofs = dof_connection->vec;

  if(m_parametric->not_all) {
    m_t_coords = get_lagrange_coord_flags(master);
    s_t_coords = get_lagrange_coord_flags(slave);

    m_t_coord_vec = m_t_coords->vec;
    s_t_coord_vec = s_t_coords->vec;

    FOR_ALL_DOFS(s_coords->fe_space->admin,
		 COPY_DOW(m_coord_vec[master_dofs[dof]], s_coord_vec[dof]);
		 s_t_coord_vec[dof] = m_t_coord_vec[master_dofs[dof]];
		 );
  }
  else {
    FOR_ALL_DOFS(s_coords->fe_space->admin,
		 COPY_DOW(m_coord_vec[master_dofs[dof]], s_coord_vec[dof]);
		 );
  }


  free_dof_int_vec(dof_connection);

  return;
}


/****************************************************************************/
/* use_lagrange_parametric(mesh, degree, n_proj, strategy):                 */
/* Transforms the mesh to use parametric simplices of degree "degree".      */
/* All mesh elements which are affected by the node projection "n_proj"     */
/* will be deformed. The parameter "strategy" determines how elements will  */
/* become parametric. The following choices are available at the moment:    */
/*                                                                          */
/*  strategy == 0: all elements of the mesh will be treated as parametric   */
/*                 elements, implying that determinants and Jacobeans will  */
/*                 be calculated at all quadrature points during assembly.  */
/*                 This is useful e.g. for triangulations of embedded       */
/*                 curved manifolds. Please note that during refinement a   */
/*                 parent element will be split along the surface defined   */
/*                 by the following equation                                */
/*                                                                          */
/*                 lambda_0 = lambda_1                                      */
/*                                                                          */
/*                 defined in the nonlinear barycentric coords lambda_i.    */
/*                 For some meshes, this will create better shaped          */
/*                 simplices.                                               */
/*                                                                          */
/*  strategy == 1: only those elements of the mesh affected by "n_proj"     */
/*                 will be treated as parametric elements. This enables     */
/*                 some optimization during assembly. All children of the   */
/*                 original curved simplices will be determined as above,   */
/*                 implying that they too will be parametric.               */
/*                                                                          */
/*  strategy == 2: only those elements of the mesh affected by "n_proj"     */
/*                 will be treated as parametric elements. The children of  */
/*                 parametric elements will only be parametric elements     */
/*                 if they too are affected by "n_proj". Parent elements    */
/*                 are split along straight lines or planes, which could    */
/*                 lead to degenerate elements. The purpose of this         */
/*                 algorithm is to keep the number of curved elements as    */
/*                 small as possible. This is useful if one only wishes to  */
/*                 use parametric elements to approximate a curved boundary */
/*                 or interface manifold.                                   */
/****************************************************************************/

void use_lagrange_parametric(MESH *mesh, int degree,
			     NODE_PROJECTION *n_proj, int strategy)
{
  FUNCNAME("use_lagrange_parametric");
  PARAMETRIC     *parametric;
  DOF_REAL_D_VEC *coords;
  DOF_UCHAR_VEC  *touched_coords = nil;
  REAL_D         *local_coords;
  const BAS_FCTS *lagrange = nil;
  int             dim, i;
  MESH            *slave;
  const FE_SPACE  *fe_space;
  const FE_SPACE  *s_fe_space;
  const BAS_FCTS  *s_bas_fcts;

  TEST_EXIT(mesh,"No fe_space given!\n");

  TEST_EXIT(!mesh->parametric,
    "There is already a parametric structure defined on this mesh!\n");

  dim    = mesh->dim;

  TEST_EXIT((dim > 0) && (dim <= DIM_OF_WORLD),
    "Parametric elements of dimension %d are not available for DIM_OF_WORLD == %d!\n", dim, DIM_OF_WORLD);

  TEST_EXIT((degree > 0) && (degree < MAX_DEG+1),
    "Only implemented for 0 < degree < %d.\n", MAX_DEG+1);

  TEST_EXIT(strategy >=0 && strategy <= 2,
    "Only strategy 0,1,2 are implemented!\n");

  lagrange = get_lagrange(dim, degree);
  fe_space = get_fe_space(mesh, "main space of parametric coords", nil,
			  lagrange, false);

  local_coords = MEM_CALLOC(fe_space->bas_fcts->n_bas_fcts, REAL_D);

  coords = get_dof_real_d_vec("Lagrange parametric coordinates", fe_space);

  if(strategy > 0) {
    touched_coords = get_dof_uchar_vec("Touched parametric vertices", 
				       fe_space);
    FOR_ALL_DOFS(fe_space->admin, touched_coords->vec[dof] = 0);
  }

  coords->refine_interpol = all_refine_interpol[(dim-1)*MAX_DEG + degree - 1];
  coords->coarse_restrict = all_coarse_interpol[(dim-1)*MAX_DEG + degree - 1];

  parametric = MEM_CALLOC(1, PARAMETRIC);

  *parametric = *all_lagrange[(dim-1) * MAX_DEG + degree - 1];
  
  parametric->data = MEM_ALLOC(1, LAGRANGE_PARAM_DATA);
  ((LAGRANGE_PARAM_DATA *)parametric->data)->n_proj = n_proj;
  ((LAGRANGE_PARAM_DATA *)parametric->data)->coords = coords; 
  ((LAGRANGE_PARAM_DATA *)parametric->data)->touched_coords = touched_coords; 
  ((LAGRANGE_PARAM_DATA *)parametric->data)->n_local_coords = 
    fe_space->bas_fcts->n_bas_fcts; 
  ((LAGRANGE_PARAM_DATA *)parametric->data)->local_coords = local_coords; 

  (all_fill_coords[(dim-1)*MAX_DEG + degree - 1])
    ((LAGRANGE_PARAM_DATA *)parametric->data);

  parametric->not_all = strategy;

  /* Set inheritance function. */
  parametric->inherit_parametric = inherit_lagrange_parametric;

  mesh->parametric = parametric;

  /* Now handle submeshes... */

  if(mesh->dim > 1) {
    for(i = 0; i < ((MESH_MEM_INFO *)mesh->mem_info)->n_slaves; i++) {
      slave = ((MESH_MEM_INFO *)mesh->mem_info)->slaves[i];

      s_bas_fcts = get_lagrange(slave->dim, degree);    
      s_fe_space = get_fe_space(slave, "slave space of parametric coords",
				nil, s_bas_fcts, false);

      use_lagrange_parametric(slave, degree, n_proj, strategy);
    }
  }

  return;
}


/****************************************************************************/
/* get_lagrange_coords(mesh): Return the DOF_REAL_D_VEC coordinate vector   */
/* used to defined the Lagrange type parametric elements implemented above. */
/* USE WITH CARE, ESPECIALLY IF YOU ARE CHANGING VALUES! Problems may arise */
/* because elements are not recognized as curved elements unless the        */
/* internally used "touched_coords" vector is also set.                     */
/****************************************************************************/

DOF_REAL_D_VEC *get_lagrange_coords(MESH *mesh)
{
  FUNCNAME("get_lagrange_coords");

  TEST_EXIT(mesh, "No mesh given!\n");
  TEST_EXIT(mesh->parametric, "This is not a parametric mesh!\n");
  TEST_EXIT(mesh->parametric->name, "Can not identify parametric type!\n");
  TEST_EXIT(strstr(mesh->parametric->name, "Lagrange"),
	    "This is not a Lagrange type PARAMETRIC!\n");

  return ((LAGRANGE_PARAM_DATA *)mesh->parametric->data)->coords;
}


DOF_UCHAR_VEC *get_lagrange_coord_flags(MESH *mesh)
{
  FUNCNAME("get_lagrange_coord_flags");

  TEST_EXIT(mesh, "No mesh given!\n");
  TEST_EXIT(mesh->parametric, "This is not a parametric mesh!\n");
#if 0
  TEST_EXIT(mesh->parametric->name, "Can not identify parametric type!\n");
  TEST_EXIT(strstr(mesh->parametric->name, "Lagrange"),
            "This is not a Lagrange type PARAMETRIC!\n");
#endif
  if(mesh->parametric->not_all == 0)
    WARNING("Strategy 0 used; coordinate flag vector is not used!\n");
  
  return ((LAGRANGE_PARAM_DATA *)mesh->parametric->data)->touched_coords;
}
